# scripts/export_kernels.py
import os, json, hashlib
from pathlib import Path
import numpy as np
from scipy.interpolate import CubicSpline  # if missing: pip install scipy

# === EDIT THESE PATHS IF YOUR TREE DIFFERS ===
SPECTRUM = "data/volume3_master_spectrum.npy"     # or .csv
PIVOT_FIT = "artifacts/pivot_fit.json"
OUTDIR    = "artifacts/kernels"
GAUGES    = ["SU3", "SU2"]                         # keep one or both
L_LIST    = [128, 256]

def sha256(path: str) -> str:
    h = hashlib.sha256()
    with open(path, "rb") as f:
        for chunk in iter(lambda: f.read(1<<20), b""):
            h.update(chunk)
    return h.hexdigest()

def load_spectrum(path: str) -> np.ndarray:
    p = Path(path)
    if p.suffix == ".npy":
        return np.load(p)
    # assume csv
    return np.loadtxt(p, delimiter=",")

def resample_kernel(rho: np.ndarray, L: int) -> np.ndarray:
    x = np.linspace(0, 1, len(rho))
    cs = CubicSpline(x, rho, bc_type=("clamped", "clamped"))
    kvec = cs(np.linspace(0, 1, 2*L*L)).astype(np.float64)
    return np.maximum(kvec, 0.0)  # enforce non-negativity

def main():
    rho = load_spectrum(SPECTRUM)
    spectrum_hash = sha256(SPECTRUM) if Path(SPECTRUM).exists() else ""
    pivot_hash    = sha256(PIVOT_FIT) if Path(PIVOT_FIT).exists() else ""
    for gauge in GAUGES:
        outdir = Path(OUTDIR) / gauge
        outdir.mkdir(parents=True, exist_ok=True)
        for L in L_LIST:
            kvec = resample_kernel(rho, L)
            assert kvec.size == 2*L*L, f"Bad length for L={L}"
            np.save(outdir / f"kernel_L{L}.npy", kvec)
            meta = {
                "gauge": gauge, "L": int(L), "length": int(kvec.size),
                "spectrum_path": SPECTRUM, "pivot_fit_path": PIVOT_FIT,
                "spectrum_sha256": spectrum_hash, "pivot_fit_sha256": pivot_hash,
            }
            with open(outdir / f"kernel_L{L}_METADATA.json", "w") as f:
                json.dump(meta, f, indent=2)
            print(f"[OK] {gauge} L={L} → {outdir/f'kernel_L{L}.npy'} (len={kvec.size})")
    print("Done.")

if __name__ == "__main__":
    main()
